import { NextRequest, NextResponse } from "next/server";
import { PrismaClient } from "@prisma/client";
import { verifyPassword } from "@/lib/shorty";

const prisma = new PrismaClient();

export async function GET(req: NextRequest, { params }: { params: { slug: string } }) {
  const link = await prisma.link.findUnique({ where: { slug: params.slug } });
  if (!link) return NextResponse.json({ error: "Not found" }, { status: 404 });

  if (link.expireAt && link.expireAt.getTime() <= Date.now()) {
    return NextResponse.json({ error: "Link expired" }, { status: 410 });
  }

  if (link.passwordHash) {
    const urlPass = req.nextUrl.searchParams.get("p");
    const auth = req.headers.get("authorization")?.replace(/^Bearer /i, "");
    const supplied = urlPass || auth || "";
    const ok = await verifyPassword(supplied, link.passwordHash);
    if (!ok) {
      return new NextResponse(
        `<!doctype html><meta charset="utf-8">
         <title>Protected Link</title>
         <form method="GET" style="display:grid;gap:8px;max-width:320px;margin:40px auto;font-family:system-ui">
           <input name="p" type="password" placeholder="Password" style="padding:10px;border:1px solid #ddd;border-radius:8px" />
           <button type="submit" style="padding:10px;border-radius:8px;background:black;color:white">Unlock</button>
         </form>`,
        { status: 401, headers: { "content-type": "text/html; charset=utf-8" } }
      );
    }
  }

  await prisma.link.update({
    where: { slug: link.slug },
    data: { clicks: { increment: 1 }, ...(link.oneTime ? { expireAt: new Date() } : {}) }
  });

  return NextResponse.redirect(link.target, { status: 302 });
}
